using System;
using System.ComponentModel;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Xml.Serialization;
using Microsoft.SharePoint;
using Microsoft.SharePoint.Utilities;
using Microsoft.SharePoint.WebPartPages;

using System.Xml;

namespace EasyTabParts
{
	/// <summary>
	/// Description for ZoneTabs.
	/// </summary>
	[DefaultProperty("ZoneTabXml"),
		ToolboxData("<{0}:ZoneTabs runat=server></{0}:ZoneTabs>"),
		XmlRoot(Namespace="EasyTabParts")]
	public class ZoneTabs : Microsoft.SharePoint.WebPartPages.WebPart
	{

		#region Tool Part Attributes

		// ZoneTabXml - Holds the list of tabs and web parts hidden by each
		private const string defaultZoneTabXml = "<tabs />";
		private string zoneTabXml = defaultZoneTabXml;
		[Browsable(false),
			Category("Miscellaneous"),
			DefaultValue(defaultZoneTabXml),
			WebPartStorage(Storage.Personal),
			FriendlyName("Zone Tab XML"),
			Description("Zone tab settings")]
		public string ZoneTabXml
		{
			get
			{
				return zoneTabXml;
			}

			set
			{
				zoneTabXml = value;
			}
		}

		// TopBarPixels - Holds the # of pixels thick the top bar should be
		private const int defaultTopBarPixels = 0;
		private int topBarPixels = defaultTopBarPixels;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultTopBarPixels),
		WebPartStorage(Storage.Personal),
		FriendlyName("Top Bar"),
		Description("Top Bar Thickness in Pixels")]
		public int TopBarPixels
		{
			get
			{
				return topBarPixels;
			}

			set
			{
				topBarPixels = value;
			}
		}

		// BottomBarPixels - Holds the number of pixels thick the bottom bar
		// should be
		private const int defaultBottomBarPixels = 10;
		private int bottomBarPixels = defaultBottomBarPixels;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultBottomBarPixels),
		WebPartStorage(Storage.Personal),
		FriendlyName("Bottom Bar"),
		Description("Bottom Bar Thickness in Pixels")]
		public int BottomBarPixels
		{
			get
			{
				return bottomBarPixels;
			}

			set
			{
				bottomBarPixels = value;
			}
		}

		#endregion

		#region Tool Part Registration and Control Access

		/// <summary>
		///	This method gets the custom tool parts for this Web Part by overriding the
		///	GetToolParts method of the WebPart base class. You must implement
		///	custom tool parts in a separate class that derives from 
		///	Microsoft.SharePoint.WebPartPages.ToolPart. 
		///	</summary>
		///<returns>An array of references to ToolPart objects.</returns>
		public override ToolPart[] GetToolParts()
		{
			ToolPart[] toolparts = new ToolPart[3];
			
			// Our own tool part
			ZoneTabsToolPart zttp = new ZoneTabsToolPart();
			toolparts[0] = zttp;

			// Base tool parts
			WebPartToolPart wptp = new WebPartToolPart();
			toolparts[1] = wptp;

			// Tool parts discovered by reflection
			CustomPropertyToolPart custom = new CustomPropertyToolPart();
			toolparts[2] = custom;

			return toolparts;
		}

		// This utility function is available to the ToolPart so it can enumerate the web
		// parts in the current zone
		public WebPartZone GetZone ()
		{
			try
			{
				WebPartZone myZone;
				myZone = this.Page.FindControl (this.ZoneID)
					as WebPartZone;
				return (myZone);
			}
			catch
			{
				return (null);
			}
		}
	
		#endregion

		#region Attributes controlled by DWP only

		// LeaveEmptySpace - true if we should 1 tab width blank at the right
		private const bool defaultLeaveEmptySpace = true;
		private bool leaveEmptySpace = defaultLeaveEmptySpace;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultLeaveEmptySpace),
		WebPartStorage(Storage.Shared),
		FriendlyName("Leave Empty Space"),
		Description("Leaves empty space to the right of the last tab")]
		public bool LeaveEmptySpace
		{
			get
			{
				return leaveEmptySpace;
			}

			set
			{
				leaveEmptySpace = value;
			}
		}

		// SelectedTabCss - Holds the CSS class name for selected tabs
		private const string defaultSelectedTabCss = "ms-tabselected";
		private string selectedTabCss = defaultSelectedTabCss;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultSelectedTabCss),
		WebPartStorage(Storage.Shared),
		FriendlyName("Selected Tab CSS"),
		Description("CSS class for selected tabs")]
		public string SelectedTabCss
		{
			get
			{
				return selectedTabCss;
			}

			set
			{
				selectedTabCss = value;
			}
		}

		// UnselectedTabCss - Holds the CSS class name for selected tabs
		private const string defaultUnselectedTabCss = "ms-tabinactive";
		private string unselectedTabCss = defaultUnselectedTabCss;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultUnselectedTabCss),
		WebPartStorage(Storage.Shared),
		FriendlyName("Un-selected Tab CSS"),
		Description("CSS class for inactive tabs")]
		public string UnselectedTabCss
		{
			get
			{
				return unselectedTabCss;
			}

			set
			{
				unselectedTabCss = value;
			}
		}

		// TabBarCss - Holds the CSS class name bars above and below the tabs
		private const string defaultTabBarCss = "ms-tabselected";
		private string tabBarCss = defaultTabBarCss;
		[Browsable(false),
		Category("Miscellaneous"),
		DefaultValue(defaultTabBarCss),
		WebPartStorage(Storage.Shared),
		FriendlyName("Tab Bar CSS"),
		Description("CSS class bars above and below the tabs")]
		public string TabBarCss
		{
			get
			{
				return tabBarCss;
			}

			set
			{
				tabBarCss = value;
			}
		}

		#endregion

		#region Web Part Rendering

		// Exception Handling
		bool errorSet = false;
		string errorMessage = "";
		private void HandleException (Exception ex)
		{
			errorSet = true;
			errorMessage = ex.Message;
		}

		// This table will contain the tabs
		Table tabsTable;

		// Remember the first tab name so we don't have to hunt it down
		// each time
		string firstTabName = "";

		// CreateChildControls() - called early in the rendering process
		// by EnsureChildControls()
		protected override void CreateChildControls()
		{
			base.CreateChildControls ();

			// Create the tabs table control
			tabsTable = new Table();
			int tabCount = 0;

			try
			{
				TableRow tr;

				// Add top border of tab strip unless it's 0 pixels thick
				TableCell topBarCell = null;	// Scope so we can fix ColumnSpan later...
				if (topBarPixels > 0)
				{
					tr = new TableRow();
					topBarCell = new TableCell ();
					topBarCell.CssClass = this.TabBarCss;
					topBarCell.Height = Unit.Pixel (topBarPixels);
					tr.Cells.Add (topBarCell);
					tabsTable.Rows.Add (tr);
				}

				// This row will hold the tabs themselves
				tr = new TableRow();

				// Get the tab data into an XmlDocument, and make a node list of the tab
				// elements
				XmlDocument tabConfigDoc = new XmlDocument ();
				tabConfigDoc.LoadXml (zoneTabXml);
				XmlNodeList tabNodes = 
					tabConfigDoc.GetElementsByTagName (ZoneTabConstants.XmlElementNameTab);

				// Cacluclate the tab width
				tabCount = tabNodes.Count;
				Unit tabWidth;
				if (this.LeaveEmptySpace || tabCount == 0)
				{
					tabWidth = Unit.Percentage ((1.0/(tabCount+1.0))*100.0);
				}
				else
				{
					tabWidth = Unit.Percentage ((1.0/tabCount)*100.0);
				}

				// Now loop through the nodes adding tabs as we go
				bool isFirstTab = true;
				foreach (XmlNode node in tabNodes)
				{
					XmlElement elt = node as XmlElement;
					if (elt != null)
					{
						string tabName = 
							elt.Attributes[ZoneTabConstants.XmlAttributeNameName].Value;
						AddTab (tr, tabName, tabName, tabWidth);
						if (isFirstTab)
						{
							isFirstTab = false;
							firstTabName = tabName;
						}
					}
				}

				// If selected, add empty space at right of right tab
				if (leaveEmptySpace)
				{
					TableCell tc = new TableCell ();
					tc.Text = "&nbsp;";
					tc.ID = "blank";
					tr.Cells.Add (tc);
					tabCount++;
				}
				tabsTable.Rows.Add (tr);

				// Now that we know how many tabs we have, fix the top bar ColumnSpan
				if (topBarPixels > 0 && topBarCell != null)
				{
					topBarCell.ColumnSpan = tabCount;
				}

				// Add bottom border of tab strip, unless it's 0 pixels thick
				if (bottomBarPixels > 0)
				{
					tr = new TableRow();
					TableCell tc = new TableCell ();
					tc.ColumnSpan = tabCount;
					tc.CssClass = this.TabBarCss;
					tc.Height = Unit.Pixel (bottomBarPixels);
					tr.Cells.Add (tc);
					tabsTable.Rows.Add (tr);
				}
			}
			catch (Exception ex)
			{
				HandleException (ex);
			}
			tabsTable.Width = Unit.Percentage (100);

			this.Controls.Add (tabsTable);
		}

		// AddTab (tr, tabID, tabText) - Adds a single tab to the tabs table row
		private TableCell AddTab (TableRow tr, string tabID, string tabText)
		{
			// Make a cell and link button for the tab
			TableCell tc = new TableCell();
			LinkButton link = new LinkButton ();

			// Set up the link button, including its event handler
			link.Click +=new EventHandler(link_Click);
			link.Text = tabText;
			link.ID = tabID;

			// Set up the table cell
			tc.ID = "cell-" + tabID;
			tc.Controls.Add (link);
			tc.Height = Unit.Pixel (25);
			tc.HorizontalAlign = HorizontalAlign.Center;

			tr.Cells.Add (tc);

			return (tc);
		}

		// AddTab (tr, tabID, tabText, width) - Overload for AddTab which allows explicit
		// setting of tab width
		private TableCell  AddTab (TableRow tr, string tabID, string tabText, Unit tabWidth)
		{
			TableCell tc = AddTab (tr, tabID, tabText);
			tc.Width = tabWidth;

			return (tc);
		}


		// OnPreRender - Set the tab state as needed - this has to happen after
		// viewstate has had a chance to load and CreateChildControls has been
		// called
		protected override void OnPreRender(EventArgs e)
		{
			base.OnPreRender (e);
			if (this.BrowserDesignMode)
			{
				// We are in design mode: force show all web parts
				SetTabVisual ();
				ShowAllWebParts();
			}
			else
			{
				if (!this.Page.IsPostBack || TabsInDesignMode())
				{
					// We are rendering the 1st time or recovering from design
					// mode: auto-select 1st tab
					SetTabVisual (firstTabName);
					SetWebParts (firstTabName);
				}
			}
		}

		// linkClick (sender, e) - Event handler for clicking a tab
		private void link_Click(object sender, EventArgs e)
		{
			try
			{
				// Get the link button that fired the event
				LinkButton link = sender as LinkButton;
				if (link != null)
				{
					// Update the tab appearance and web part state
					// for this tab
					string tabName = link.ID;
					SetTabVisual (tabName);
					SetWebParts (tabName);
				}
			}
			catch (Exception ex)
			{
				errorSet = true;
				errorMessage = ex.Message;
			}
		}

		// SetTabVisual () - Called in design mode when all tabs are selected
		private void SetTabVisual ()
		{
			// Figure out which row has the tabs in it
			int tabRowIndex = 0;
			if (topBarPixels > 0)
			{
				tabRowIndex = 1;
			}

			// Set them all to selelected
			foreach (TableCell tc in tabsTable.Rows[tabRowIndex].Cells)
			{
				if (tc.ID != "blank")
				{
					tc.CssClass = this.SelectedTabCss;
				}
			}
		}

		// TabsInDesignMode() - Returns true if tabs are all selected due to having
		// been in design mode
		private bool TabsInDesignMode()
		{
			// Figure out which row has the tabs in it
			int tabRowIndex = 0;
			if (topBarPixels > 0)
			{
				tabRowIndex = 1;
			}

			// Count the selected tabs - multiple tabs mean we've been in design mode
			int selectedTabCount = 0;
			foreach (TableCell tc in tabsTable.Rows[tabRowIndex].Cells)
			{
				if (tc.ID != "blank" && tc.CssClass == this.SelectedTabCss)
				{
					selectedTabCount++;
				}
			}
			return (selectedTabCount > 1);
		}

		// SetTabVisual (tabName) - Called to set a particular tab by name
		private void SetTabVisual (string tabName)
		{
			// Figure out which row has the tabs in it
			int tabRowIndex = 0;
			if (topBarPixels > 0)
			{
				tabRowIndex = 1;
			}

			// Scan the cells and set only the selected tab
			foreach (TableCell tc in tabsTable.Rows[tabRowIndex].Cells)
			{
				if (tc.ID != "blank")
				{
					if (tc.ID == "cell-" + tabName)
					{
						tc.CssClass = this.SelectedTabCss;
					}
					else
					{
						tc.CssClass = this.UnselectedTabCss;
					}
				}
			}
		}

		// SetWebParts (tabName) - Set web part state for the specified tab
		private void SetWebParts (string tabName)
		{
			// First, build an XPath query to find the <tab> element for the
			// specified tab
			string xpathQuery = "//" +
				ZoneTabConstants.XmlElementNameTab +
				"[@" +
				ZoneTabConstants.XmlAttributeNameName +
				"=\"" + tabName + "\"]";

			try
			{
				// Now get the Xml and  locate the element
				XmlDocument tabConfigDoc = new XmlDocument ();
				tabConfigDoc.LoadXml (zoneTabXml);
				XmlElement tabConfigElement = 
					tabConfigDoc.SelectSingleNode (xpathQuery) as XmlElement;

				if (tabConfigElement != null)
				{
					// If here, we successfully found the <tab> element we're
					// looking for.
					// Now scan the current web part zone and examine each web
					// part
					WebPartZone myZone;
					myZone = this.Page.FindControl (this.ZoneID)
						as WebPartZone;

					if (myZone != null)
					{
						for (int i=0; i < myZone.Controls.Count; i++)
						{
							// Get the web part
							WebPart wp = myZone.Controls [i] as WebPart;
							if (wp != null)
							{
								// Build an XPath query to get the attribute for
								// this web part
								xpathQuery = "./" +
									ZoneTabConstants.XmlElementNameWebPart +
									"[@" +
									ZoneTabConstants.XmlAttributeTitleName +
									"=\"" + wp.Title + "\"]";
								XmlElement wpElement = 
									tabConfigElement.SelectSingleNode (xpathQuery) as XmlElement;
								if (wpElement != null)
								{
									// If here, we found an XML element for this web part.
									// If it's supposed to be hidden, then hide it!
									XmlAttribute visibleAttribute = wpElement.Attributes["visible"];
									if (visibleAttribute != null &&
										!this.BrowserDesignMode &&
										visibleAttribute.Value == 
											ZoneTabConstants.XmlAttributeVisibleValueFalse)
									{
										// Hide the web part
										wp.Visible = false;
									}
									else
									{
										//Show the web part
										wp.Visible = true;
									}
								}
								else
								{
									// If here, there was no XML element for this web part.
									// Show it!
									wp.Visible = true;
								}
							}
						}
					}
				}
			}
			catch (Exception ex)
			{
				HandleException (ex);
			}
		}

		// ShowAllWebParts () - Turns all the web parts on, for use in design mode
		private void ShowAllWebParts ()
		{
			try
			{
				// Get the web part zone
				WebPartZone myZone;
				myZone = this.Page.FindControl (this.ZoneID)
					as WebPartZone;

				if (myZone != null)
				{
					// Loop through the contained web parts and turn them all to
					// visible
					for (int i=0; i < myZone.Controls.Count; i++)
					{
						WebPart wp = myZone.Controls [i] as WebPart;
						if (wp != null)
						{
							wp.Visible = true;
						}
					}
				}
			}
			catch (Exception ex)
			{
				HandleException (ex);
			}
		}

		/// <summary>
		/// Render this Web Part to the output parameter specified.
		/// </summary>
		/// <param name="output"> The HTML writer to write out to </param>
		protected override void RenderWebPart(HtmlTextWriter output)
		{
			if (!errorSet)
			{
				base.RenderWebPart (output);
			}
			else
			{
				output.Write(SPEncode.HtmlEncode(errorMessage));
			}
		}

		#endregion

	}
}
